<?php
$GLOBALS['path_depth'] = 1;
require_once '../../config/database.php';

// Ensure user is logged in and is a super_admin
if (!isset($_SESSION['admin_id']) || ($_SESSION['role'] ?? '') !== 'super') {
    header("Location: ../admin_login.php");
    exit;
}

// Ensure game_id column exists before processing actions
$checkColumn = $pdo->query("SHOW COLUMNS FROM admins LIKE 'game_id'");
if (!$checkColumn->fetch()) {
    $pdo->exec("ALTER TABLE admins ADD COLUMN game_id INT DEFAULT NULL");
}

$checkAssigned = $pdo->query("SHOW COLUMNS FROM admins LIKE 'assigned_game_id'");
if (!$checkAssigned->fetch()) {
    $pdo->exec("ALTER TABLE admins ADD COLUMN assigned_game_id INT DEFAULT NULL");
}

$message = '';
$error = '';

// Handle Form Submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            $username = trim($_POST['username']);
            $password = $_POST['password'];
            $game_id = !empty($_POST['game_id']) ? $_POST['game_id'] : null;

            if (empty($username) || empty($password)) {
                $error = "Username and Password are required.";
            } else {
                // Check if username exists
                $stmt = $pdo->prepare("SELECT id FROM admins WHERE username = ?");
                $stmt->execute([$username]);
                if ($stmt->fetch()) {
                    $error = "Username already exists.";
                } else {
                    $hash = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("INSERT INTO admins (username, password_hash, role, game_id) VALUES (?, ?, 'admin', ?)");
                    if ($stmt->execute([$username, $hash, $game_id])) {
                        $message = "Controller added successfully.";
                    } else {
                        $error = "Failed to add controller.";
                    }
                }
            }
        } elseif ($_POST['action'] === 'edit') {
            $id = $_POST['id'];
            $username = trim($_POST['username']);
            $password = $_POST['password'];
            $game_id = !empty($_POST['game_id']) ? $_POST['game_id'] : null;

            if (empty($username)) {
                $error = "Username is required.";
            } else {
                // Check if username exists for other users
                $stmt = $pdo->prepare("SELECT id FROM admins WHERE username = ? AND id != ?");
                $stmt->execute([$username, $id]);
                if ($stmt->fetch()) {
                    $error = "Username already exists.";
                } else {
                    if (!empty($password)) {
                        $hash = password_hash($password, PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("UPDATE admins SET username = ?, password_hash = ?, game_id = ? WHERE id = ? AND role = 'admin'");
                        $stmt->execute([$username, $hash, $game_id, $id]);
                    } else {
                        $stmt = $pdo->prepare("UPDATE admins SET username = ?, game_id = ? WHERE id = ? AND role = 'admin'");
                        $stmt->execute([$username, $game_id, $id]);
                    }
                    $message = "Controller updated successfully.";
                }
            }
        } elseif ($_POST['action'] === 'delete') {
            $id = $_POST['id'];
            $stmt = $pdo->prepare("DELETE FROM admins WHERE id = ? AND role = 'admin'");
            if ($stmt->execute([$id])) {
                $message = "Controller deleted successfully.";
            } else {
                $error = "Failed to delete controller.";
            }
        }
    }
}

// Fetch Controllers
$controllers = $pdo->query("SELECT a.*, g.game_name, COALESCE(a.game_id, a.assigned_game_id) as effective_game_id FROM admins a LEFT JOIN games g ON COALESCE(a.game_id, a.assigned_game_id) = g.id WHERE a.role = 'admin' ORDER BY a.username ASC")->fetchAll();

// Fetch Games for Dropdown
$games = $pdo->query("SELECT * FROM games ORDER BY game_name ASC")->fetchAll();
?>
<!DOCTYPE html>
<html class="dark" lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>MAGMA Manage Controllers</title>
    <link href="https://fonts.googleapis.com/css2?family=Spline+Sans:wght@300;400;500;600;700&display=swap" rel="stylesheet"/>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:wght,FILL@100..700,0..1&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <script>
        tailwind.config = {
            darkMode: "class",
            theme: {
                extend: {
                    colors: {
                        "primary": "#ff1a1a",
                        "background-dark": "#0a0a0a",
                        "surface-dark": "#121212",
                    },
                    fontFamily: {
                        "display": ["Spline Sans", "sans-serif"]
                    }
                },
            },
        }
    </script>
    <link rel="stylesheet" href="../css/admin_dashboard.css">
</head>
<body class="bg-background-dark text-white font-display flex h-screen overflow-hidden">
    
    <!-- Sidebar -->
    <aside class="w-64 bg-surface-dark border-r border-white/5 flex flex-col z-20 hidden md:flex">
        <div class="p-6 flex items-center gap-3 border-b border-white/5">
            <div class="w-10 h-10 rounded-lg bg-primary/20 flex items-center justify-center border border-primary/30">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
            </div>
            <div>
                <h1 class="font-bold text-lg tracking-wider">MAGMA</h1>
                <p class="text-[10px] text-gray-500 uppercase tracking-widest font-bold">Super Admin</p>
            </div>
        </div>
        
        <nav class="flex-1 py-6 space-y-1">
            <a href="dashboard.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">dashboard</span>
                Dashboard
            </a>
            <a href="controllers.php" class="sidebar-link active flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">sports_esports</span>
                Manage Controllers
            </a>
            <a href="games.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">videogame_asset</span>
                Games & Scoring
            </a>
            <a href="participants.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">groups</span>
                Participants
            </a>
            <a href="requests.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">pending_actions</span>
                Requests
            </a>
            <a href="logs.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">receipt_long</span>
                System Logs
            </a>
        </nav>

        <div class="p-4 border-t border-white/5">
            <a href="logout.php" class="flex items-center gap-3 px-4 py-3 text-sm font-medium text-red-400 hover:bg-red-500/10 rounded-lg transition-colors">
                <span class="material-symbols-outlined">logout</span>
                Sign Out
            </a>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 flex flex-col relative overflow-hidden">
        <div class="absolute inset-0 geo-overlay z-0"></div>
        
        <!-- Mobile Header -->
        <header class="md:hidden h-16 border-b border-white/5 flex items-center justify-between px-4 bg-surface-dark z-20">
            <div class="flex items-center gap-2">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
                <span class="font-bold tracking-wider">MAGMA</span>
            </div>
            <button class="text-gray-400 hover:text-white">
                <span class="material-symbols-outlined">menu</span>
            </button>
        </header>

        <div class="flex-1 overflow-y-auto p-6 md:p-10 relative z-10 custom-scrollbar">
            <div class="max-w-5xl mx-auto">
                <div class="flex items-end justify-between mb-8">
                    <div>
                        <h2 class="text-3xl font-bold text-white mb-1">Game Controllers</h2>
                        <p class="text-gray-500 text-sm">Manage access for game agents.</p>
                    </div>
                    <button onclick="document.getElementById('addModal').classList.remove('hidden')" class="bg-primary hover:bg-red-600 text-white px-4 py-2 rounded-lg font-bold text-sm uppercase tracking-wider flex items-center gap-2 transition-colors shadow-lg shadow-primary/20">
                        <span class="material-symbols-outlined">add</span>
                        Add New
                    </button>
                </div>

                <!-- Controllers List -->
                <div class="glass-panel rounded-2xl overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr class="bg-white/5 border-b border-white/5">
                                    <th class="p-4 pl-6 text-xs font-bold text-gray-500 uppercase tracking-wider">Username</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider">Assigned Game</th>
                                    <th class="p-4 text-xs font-bold text-gray-500 uppercase tracking-wider text-right pr-6">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="divide-y divide-white/5">
                                <?php foreach ($controllers as $controller): ?>
                                <tr class="hover:bg-white/5 transition-colors group">
                                    <td class="p-4 pl-6 font-medium text-white flex items-center gap-3">
                                        <div class="w-8 h-8 rounded-full bg-white/10 flex items-center justify-center text-gray-400">
                                            <span class="material-symbols-outlined text-lg">person</span>
                                        </div>
                                        <?php echo htmlspecialchars($controller['username']); ?>
                                    </td>
                                    <td class="p-4 text-sm text-gray-400">
                                        <?php echo htmlspecialchars($controller['game_name'] ?? 'None'); ?>
                                    </td>
                                    <td class="p-4 pr-6 text-right">
                                        <div class="flex items-center justify-end gap-2">
                                            <button 
                                                data-id="<?php echo $controller['id']; ?>" 
                                                data-username="<?php echo htmlspecialchars($controller['username']); ?>"
                                                data-game-id="<?php echo $controller['game_id'] ?? $controller['assigned_game_id'] ?? ''; ?>"
                                                onclick="openEditModal(this.dataset.id, this.dataset.username, this.dataset.gameId)" 
                                                class="p-2 rounded-lg hover:bg-white/10 text-gray-400 hover:text-white transition-colors">
                                                <span class="material-symbols-outlined text-lg">edit</span>
                                            </button>
                                            <form method="POST" onsubmit="return confirm('Are you sure you want to delete this controller?');" class="inline">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?php echo $controller['id']; ?>">
                                                <button type="submit" class="p-2 rounded-lg hover:bg-red-500/10 text-gray-400 hover:text-red-500 transition-colors">
                                                    <span class="material-symbols-outlined text-lg">delete</span>
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <?php if (empty($controllers)): ?>
                                    <tr>
                                        <td colspan="3" class="p-8 text-center text-gray-500">No controllers found. Add one to get started.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Add Modal -->
    <div id="addModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-black/80 backdrop-blur-sm" onclick="document.getElementById('addModal').classList.add('hidden')"></div>
        <div class="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-full max-w-md p-6">
            <div class="glass-panel rounded-2xl p-6 shadow-2xl border border-white/10 bg-[#1a1a1a]">
                <h3 class="text-xl font-bold text-white mb-4">Add New Controller</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="add">
                    <div class="space-y-4">
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Username</label>
                            <input type="text" name="username" required class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none">
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Assign Game (Optional)</label>
                            <select name="game_id" class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none">
                                <option value="">-- Select Game --</option>
                                <?php foreach ($games as $game): ?>
                                    <option value="<?php echo $game['id']; ?>"><?php echo htmlspecialchars($game['game_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Password</label>
                            <input type="password" name="password" required class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none">
                        </div>
                        <div class="flex gap-3 mt-6">
                            <button type="button" onclick="document.getElementById('addModal').classList.add('hidden')" class="flex-1 px-4 py-2 rounded-lg border border-white/10 text-gray-400 hover:text-white hover:bg-white/5 transition-colors font-bold text-sm">Cancel</button>
                            <button type="submit" class="flex-1 px-4 py-2 rounded-lg bg-primary hover:bg-red-600 text-white font-bold text-sm transition-colors shadow-lg shadow-primary/20">Create</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Modal -->
    <div id="editModal" class="fixed inset-0 z-50 hidden">
        <div class="absolute inset-0 bg-black/80 backdrop-blur-sm" onclick="document.getElementById('editModal').classList.add('hidden')"></div>
        <div class="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-full max-w-md p-6">
            <div class="glass-panel rounded-2xl p-6 shadow-2xl border border-white/10 bg-[#1a1a1a]">
                <h3 class="text-xl font-bold text-white mb-4">Edit Controller</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="edit">
                    <input type="hidden" name="id" id="edit_id">
                    <div class="space-y-4">
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Username</label>
                            <input type="text" name="username" id="edit_username" required class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none">
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">Assign Game (Optional)</label>
                            <select name="game_id" id="edit_game_id" class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none">
                                <option value="">-- Select Game --</option>
                                <?php foreach ($games as $game): ?>
                                    <option value="<?php echo $game['id']; ?>"><?php echo htmlspecialchars($game['game_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label class="block text-xs font-bold text-gray-500 uppercase tracking-wider mb-1">New Password (Optional)</label>
                            <input type="password" name="password" class="w-full bg-black/40 border border-white/10 rounded-lg px-4 py-2 text-white focus:border-primary focus:ring-1 focus:ring-primary outline-none" placeholder="Leave blank to keep current">
                        </div>
                        <div class="flex gap-3 mt-6">
                            <button type="button" onclick="document.getElementById('editModal').classList.add('hidden')" class="flex-1 px-4 py-2 rounded-lg border border-white/10 text-gray-400 hover:text-white hover:bg-white/5 transition-colors font-bold text-sm">Cancel</button>
                            <button type="submit" class="flex-1 px-4 py-2 rounded-lg bg-primary hover:bg-red-600 text-white font-bold text-sm transition-colors shadow-lg shadow-primary/20">Update</button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function openEditModal(id, username, gameId) {
            document.getElementById('edit_id').value = id;
            document.getElementById('edit_username').value = username;
            document.getElementById('edit_game_id').value = gameId || "";
            document.getElementById('editModal').classList.remove('hidden');
        }
    </script>

    <script src="../js/toast.js"></script>
    <?php if ($message): ?>
    <script>
        document.addEventListener('DOMContentLoaded', () => showToast("<?php echo htmlspecialchars($message); ?>", "success"));
    </script>
    <?php endif; ?>
    <?php if ($error): ?>
    <script>
        document.addEventListener('DOMContentLoaded', () => showToast("<?php echo htmlspecialchars($error); ?>", "error"));
    </script>
    <?php endif; ?>
</body>
</html>