<?php
$GLOBALS['path_depth'] = 1;
require_once '../../config/database.php';

// Ensure user is logged in and is a super_admin
if (!isset($_SESSION['admin_id']) || ($_SESSION['role'] ?? '') !== 'super') {
    header("Location: ../admin_login.php");
    exit;
}

$message = '';
$error = '';

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['request_id'])) {
        $requestId = $_POST['request_id'];
        
        // Fetch request details
        $stmt = $pdo->prepare("SELECT * FROM profile_edit_requests WHERE id = ? AND status = 'pending'");
        $stmt->execute([$requestId]);
        $request = $stmt->fetch();

        if ($request) {
            if ($_POST['action'] === 'approve') {
                try {
                    $pdo->beginTransaction();

                    // Update participant profile
                    $updateStmt = $pdo->prepare("UPDATE participants SET name = ?, college = ?, year = ?, department = ?, phone = ? WHERE id = ?");
                    $updateStmt->execute([
                        $request['new_name'],
                        $request['new_college'],
                        $request['new_year'],
                        $request['new_department'],
                        $request['new_phone'],
                        $request['participant_id']
                    ]);

                    // Update request status
                    $statusStmt = $pdo->prepare("UPDATE profile_edit_requests SET status = 'approved' WHERE id = ?");
                    $statusStmt->execute([$requestId]);

                    $pdo->commit();
                    $message = "Request approved and profile updated.";
                } catch (Exception $e) {
                    $pdo->rollBack();
                    $error = "Failed to approve request: " . $e->getMessage();
                }
            } elseif ($_POST['action'] === 'reject') {
                $stmt = $pdo->prepare("UPDATE profile_edit_requests SET status = 'rejected' WHERE id = ?");
                if ($stmt->execute([$requestId])) {
                    $message = "Request rejected.";
                } else {
                    $error = "Failed to reject request.";
                }
            }
        } else {
            $error = "Request not found or already processed.";
        }
    }
}

// Fetch Pending Requests with Participant info
$sql = "SELECT r.*, p.name as current_name, p.unique_id 
        FROM profile_edit_requests r 
        JOIN participants p ON r.participant_id = p.id 
        WHERE r.status = 'pending' 
        ORDER BY r.id ASC";
$requests = $pdo->query($sql)->fetchAll();
?>
<!DOCTYPE html>
<html class="dark" lang="en">
<head>
    <meta charset="utf-8"/>
    <meta content="width=device-width, initial-scale=1.0" name="viewport"/>
    <title>MAGMA Manage Requests</title>
    <link href="https://fonts.googleapis.com/css2?family=Spline+Sans:wght@300;400;500;600;700&display=swap" rel="stylesheet"/>
    <link href="https://fonts.googleapis.com/css2?family=Material+Symbols+Outlined:wght,FILL@100..700,0..1&display=swap" rel="stylesheet"/>
    <script src="https://cdn.tailwindcss.com?plugins=forms,container-queries"></script>
    <script>
        tailwind.config = {
            darkMode: "class",
            theme: {
                extend: {
                    colors: {
                        "primary": "#ff1a1a",
                        "background-dark": "#0a0a0a",
                        "surface-dark": "#121212",
                    },
                    fontFamily: {
                        "display": ["Spline Sans", "sans-serif"]
                    }
                },
            },
        }
    </script>
    <link rel="stylesheet" href="../css/admin_dashboard.css">
</head>
<body class="bg-background-dark text-white font-display flex h-screen overflow-hidden">
    
    <!-- Sidebar -->
    <aside class="w-64 bg-surface-dark border-r border-white/5 flex flex-col z-20 hidden md:flex">
        <div class="p-6 flex items-center gap-3 border-b border-white/5">
            <div class="w-10 h-10 rounded-lg bg-primary/20 flex items-center justify-center border border-primary/30">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
            </div>
            <div>
                <h1 class="font-bold text-lg tracking-wider">MAGMA</h1>
                <p class="text-[10px] text-gray-500 uppercase tracking-widest font-bold">Super Admin</p>
            </div>
        </div>
        
        <nav class="flex-1 py-6 space-y-1">
            <a href="dashboard.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">dashboard</span>
                Dashboard
            </a>
            <a href="controllers.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">sports_esports</span>
                Manage Controllers
            </a>
            <a href="games.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">videogame_asset</span>
                Games & Scoring
            </a>
            <a href="participants.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">groups</span>
                Participants
            </a>
            <a href="requests.php" class="sidebar-link active flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">pending_actions</span>
                Requests
            </a>
            <a href="logs.php" class="sidebar-link flex items-center gap-3 px-6 py-3 text-sm font-medium text-gray-300">
                <span class="material-symbols-outlined">receipt_long</span>
                System Logs
            </a>
        </nav>

        <div class="p-4 border-t border-white/5">
            <a href="logout.php" class="flex items-center gap-3 px-4 py-3 text-sm font-medium text-red-400 hover:bg-red-500/10 rounded-lg transition-colors">
                <span class="material-symbols-outlined">logout</span>
                Sign Out
            </a>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="flex-1 flex flex-col relative overflow-hidden">
        <div class="absolute inset-0 geo-overlay z-0"></div>
        
        <!-- Mobile Header -->
        <header class="md:hidden h-16 border-b border-white/5 flex items-center justify-between px-4 bg-surface-dark z-20">
            <div class="flex items-center gap-2">
                <span class="material-symbols-outlined text-primary">admin_panel_settings</span>
                <span class="font-bold tracking-wider">MAGMA</span>
            </div>
            <button class="text-gray-400 hover:text-white">
                <span class="material-symbols-outlined">menu</span>
            </button>
        </header>

        <div class="flex-1 overflow-y-auto p-6 md:p-10 relative z-10 custom-scrollbar">
            <div class="max-w-6xl mx-auto">
                <div class="flex items-end justify-between mb-8">
                    <div>
                        <h2 class="text-3xl font-bold text-white mb-1">Edit Requests</h2>
                        <p class="text-gray-500 text-sm">Review and approve participant profile changes.</p>
                    </div>
                </div>

                <?php if ($message): ?>
                    <div class="p-4 mb-6 text-sm text-green-400 bg-green-900/20 border border-green-500/30 rounded-xl flex items-center gap-2">
                        <span class="material-symbols-outlined">check_circle</span>
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="p-4 mb-6 text-sm text-red-400 bg-red-900/20 border border-red-500/30 rounded-xl flex items-center gap-2">
                        <span class="material-symbols-outlined">error</span>
                        <?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>

                <!-- Requests Grid -->
                <div class="grid grid-cols-1 gap-6">
                    <?php foreach ($requests as $req): ?>
                    <div class="glass-panel rounded-2xl p-6 border border-white/10">
                        <div class="flex flex-col lg:flex-row justify-between gap-6">
                            <!-- Current Info -->
                            <div class="flex-1">
                                <div class="flex items-center gap-3 mb-4">
                                    <div class="w-10 h-10 rounded-full bg-white/10 flex items-center justify-center">
                                        <span class="material-symbols-outlined text-gray-400">person</span>
                                    </div>
                                    <div>
                                        <h3 class="font-bold text-white"><?php echo htmlspecialchars($req['current_name']); ?></h3>
                                        <p class="text-xs text-gray-500 font-mono"><?php echo htmlspecialchars($req['unique_id']); ?></p>
                                    </div>
                                </div>
                            </div>

                            <!-- Changes -->
                            <div class="flex-[2] grid grid-cols-1 md:grid-cols-2 gap-4 bg-black/20 p-4 rounded-xl border border-white/5">
                                <div>
                                    <p class="text-[10px] text-gray-500 uppercase tracking-wider font-bold mb-1">New Name</p>
                                    <p class="text-sm text-white font-medium"><?php echo htmlspecialchars($req['new_name']); ?></p>
                                </div>
                                <div>
                                    <p class="text-[10px] text-gray-500 uppercase tracking-wider font-bold mb-1">New College</p>
                                    <p class="text-sm text-white font-medium"><?php echo htmlspecialchars($req['new_college']); ?></p>
                                </div>
                                <div>
                                    <p class="text-[10px] text-gray-500 uppercase tracking-wider font-bold mb-1">New Department</p>
                                    <p class="text-sm text-white font-medium"><?php echo htmlspecialchars($req['new_department']); ?></p>
                                </div>
                                <div class="flex gap-4">
                                    <div>
                                        <p class="text-[10px] text-gray-500 uppercase tracking-wider font-bold mb-1">Year</p>
                                        <p class="text-sm text-white font-medium"><?php echo htmlspecialchars($req['new_year']); ?></p>
                                    </div>
                                    <div>
                                        <p class="text-[10px] text-gray-500 uppercase tracking-wider font-bold mb-1">Phone</p>
                                        <p class="text-sm text-white font-medium"><?php echo htmlspecialchars($req['new_phone']); ?></p>
                                    </div>
                                </div>
                            </div>

                            <!-- Actions -->
                            <div class="flex flex-row lg:flex-col justify-center gap-3">
                                <form method="POST" onsubmit="return confirm('Approve this request?');">
                                    <input type="hidden" name="action" value="approve">
                                    <input type="hidden" name="request_id" value="<?php echo $req['id']; ?>">
                                    <button type="submit" class="w-full px-4 py-2 rounded-lg bg-green-500/20 hover:bg-green-500/30 text-green-400 border border-green-500/30 font-bold text-sm flex items-center justify-center gap-2 transition-colors">
                                        <span class="material-symbols-outlined text-lg">check</span>
                                        Approve
                                    </button>
                                </form>
                                <form method="POST" onsubmit="return confirm('Reject this request?');">
                                    <input type="hidden" name="action" value="reject">
                                    <input type="hidden" name="request_id" value="<?php echo $req['id']; ?>">
                                    <button type="submit" class="w-full px-4 py-2 rounded-lg bg-red-500/20 hover:bg-red-500/30 text-red-400 border border-red-500/30 font-bold text-sm flex items-center justify-center gap-2 transition-colors">
                                        <span class="material-symbols-outlined text-lg">close</span>
                                        Reject
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    
                    <?php if (empty($requests)): ?>
                        <div class="glass-panel rounded-2xl p-12 text-center">
                            <div class="w-16 h-16 rounded-full bg-white/5 flex items-center justify-center mx-auto mb-4">
                                <span class="material-symbols-outlined text-3xl text-gray-600">inbox</span>
                            </div>
                            <h3 class="text-xl font-bold text-white mb-2">No Pending Requests</h3>
                            <p class="text-gray-500">All profile edit requests have been processed.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>
</body>
</html>