<?php
require_once '../config/database.php';

// Access Control
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}
if (($_SESSION['role'] ?? '') !== 'super') {
    die('<div class="container mt-5"><div class="alert alert-danger">Access Denied. Super Admin only. <a href="admin_dashboard.php">Back</a></div></div>');
}

$checks = [];

// 1. Database Connection
if ($pdo) {
    $checks['database_connection'] = ['status' => 'ok', 'message' => 'Connected'];
} else {
    $checks['database_connection'] = ['status' => 'error', 'message' => 'Failed'];
}

// 2. Database Tables
$requiredTables = ['participants', 'admins', 'games', 'participant_points', 'admin_logs', 'profile_edit_requests'];
$tablesStatus = [];
foreach ($requiredTables as $table) {
    try {
        $result = $pdo->query("SHOW TABLES LIKE '$table'");
        if ($result->rowCount() > 0) {
            $tablesStatus[$table] = 'ok';
        } else {
            $tablesStatus[$table] = 'missing';
        }
    } catch (PDOException $e) {
        $tablesStatus[$table] = 'error';
    }
}
$checks['tables'] = $tablesStatus;

// 3. Server Info
$checks['php_version'] = phpversion();
$checks['server_software'] = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
try {
    $checks['db_version'] = $pdo->query('select version()')->fetchColumn();
} catch (Exception $e) {
    $checks['db_version'] = 'Unknown';
}

// 4. Error Logs
$logFile = ini_get('error_log');
$logContent = [];

if ($logFile && file_exists($logFile)) {
    if (is_readable($logFile)) {
        // Read the last 4KB to get recent logs
        $fp = fopen($logFile, 'r');
        $size = filesize($logFile);
        $pos = max(0, $size - 4096); 
        fseek($fp, $pos);
        $rawLogs = fread($fp, 4096);
        fclose($fp);
        
        $lines = explode("\n", trim($rawLogs));
        $logContent = array_slice($lines, -20);
    } else {
        $logContent = ["Log file exists but is not readable."];
    }
} else {
    $logContent = ["Error log file not found (Path: " . ($logFile ?: 'Not set in php.ini') . ")"];
}

include '../views/header.php';
?>

<div class="container">
    <h2 class="mb-4">System Health Check</h2>

    <div class="row g-4">
        <div class="col-md-6">
            <div class="card shadow-soft h-100">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0">Server Environment</h5>
                </div>
                <div class="card-body">
                    <ul class="list-group list-group-flush">
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            PHP Version
                            <span class="badge bg-primary rounded-pill"><?php echo htmlspecialchars($checks['php_version']); ?></span>
                        </li>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Database Version
                            <span class="badge bg-secondary rounded-pill"><?php echo htmlspecialchars($checks['db_version']); ?></span>
                        </li>
                        <li class="list-group-item">
                            <small class="text-muted">Server Software</small><br>
                            <?php echo htmlspecialchars($checks['server_software']); ?>
                        </li>
                    </ul>
                </div>
            </div>
        </div>

        <div class="col-md-6">
            <div class="card shadow-soft h-100">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0">Database Integrity</h5>
                </div>
                <div class="card-body">
                    <ul class="list-group list-group-flush">
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Connection Status
                            <?php if($checks['database_connection']['status'] == 'ok'): ?>
                                <span class="badge bg-success">Active</span>
                            <?php else: ?>
                                <span class="badge bg-danger">Failed</span>
                            <?php endif; ?>
                        </li>
                        <?php foreach($checks['tables'] as $table => $status): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            Table: <code><?php echo htmlspecialchars($table); ?></code>
                            <?php if($status == 'ok'): ?>
                                <span class="text-success fw-bold">✔ Exists</span>
                            <?php else: ?>
                                <span class="text-danger fw-bold">✘ Missing</span>
                            <?php endif; ?>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <div class="row mt-4">
        <div class="col-12">
            <div class="card shadow-soft">
                <div class="card-header bg-dark text-white">
                    <h5 class="mb-0">PHP Error Logs (Last 20 lines)</h5>
                </div>
                <div class="card-body bg-light">
                    <pre class="mb-0" style="max-height: 300px; overflow-y: auto; font-size: 0.85rem; white-space: pre-wrap;"><?php 
                        foreach ($logContent as $line) {
                            echo htmlspecialchars($line) . "\n";
                        }
                    ?></pre>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../views/footer.php'; ?>