<?php
require_once '../config/database.php';

// Access Control
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}
if ($_SESSION['role'] !== 'super') {
    die('<div class="container mt-5"><div class="alert alert-danger">Access Denied. Super Admin only. <a href="admin_dashboard.php">Back</a></div></div>');
}

$message = '';
$editMode = false;
$editGame = null;

// Handle Delete
if (isset($_GET['delete'])) {
    $id = $_GET['delete'];
    $stmt = $pdo->prepare("DELETE FROM games WHERE id = ?");
    $stmt->execute([$id]);
    header("Location: manage_games.php?msg=deleted");
    exit;
}

// Handle Edit Mode
if (isset($_GET['edit'])) {
    $editMode = true;
    $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
    $stmt->execute([$_GET['edit']]);
    $editGame = $stmt->fetch();
}

// Handle Form Submission (Add/Update)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $game_name = trim($_POST['game_name']);
    $max_points = (int)$_POST['max_points'];
    $game_id = $_POST['game_id'] ?? null;

    if (empty($game_name) || $max_points <= 0) {
        $message = "Invalid input. Name required and Max Points > 0.";
    } else {
        try {
            if ($game_id) {
                // Update
                $stmt = $pdo->prepare("UPDATE games SET game_name = ?, max_points = ? WHERE id = ?");
                $stmt->execute([$game_name, $max_points, $game_id]);
                $message = "Game updated successfully!";
                $editMode = false;
                $editGame = null;
            } else {
                // Insert
                $stmt = $pdo->prepare("INSERT INTO games (game_name, max_points) VALUES (?, ?)");
                $stmt->execute([$game_name, $max_points]);
                $message = "Game added successfully!";
            }
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $message = "Error: Game name already exists."; // Assuming unique constraint or logic check
            } else {
                $message = "Database error: " . $e->getMessage();
            }
        }
    }
}

// Fetch All Games
$games = $pdo->query("SELECT * FROM games ORDER BY created_at DESC")->fetchAll();

include '../views/header.php';
?>
<link rel="stylesheet" href="css/manage_games.css">
<div>
    <div class="row">
        <!-- Form Section -->
        <div class="col-md-4 mb-4">
            <div class="card shadow-soft">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><?php echo $editMode ? 'Edit Game' : 'Add New Game'; ?></h5>
                </div>
                <div class="card-body">
                    <?php if ($message): ?>
                        <div class="alert alert-info"><?php echo htmlspecialchars($message); ?></div>
                    <?php endif; ?>
                    
                    <form method="POST" action="manage_games.php">
                        <?php if ($editMode): ?>
                            <input type="hidden" name="game_id" value="<?php echo $editGame['id']; ?>">
                        <?php endif; ?>
                        
                        <div class="mb-3">
                            <label class="form-label">Game Name</label>
                            <input type="text" name="game_name" class="form-control" value="<?php echo $editMode ? htmlspecialchars($editGame['game_name']) : ''; ?>" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Max Points</label>
                            <input type="number" name="max_points" class="form-control" min="1" value="<?php echo $editMode ? htmlspecialchars($editGame['max_points']) : ''; ?>" required>
                        </div>
                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-success"><?php echo $editMode ? 'Update Game' : 'Add Game'; ?></button>
                            <?php if ($editMode): ?>
                                <a href="manage_games.php" class="btn btn-secondary">Cancel</a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- List Section -->
        <div class="col-md-8">
            <div class="table-container shadow-soft">
                <h4 class="mb-3">Existing Games</h4>
                <table class="table table-hover">
                    <thead class="table-light">
                        <tr>
                            <th>ID</th>
                            <th>Game Name</th>
                            <th>Max Points</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($games as $game): ?>
                        <tr>
                            <td><?php echo $game['id']; ?></td>
                            <td><?php echo htmlspecialchars($game['game_name']); ?></td>
                            <td><span class="badge bg-info text-dark"><?php echo $game['max_points']; ?> pts</span></td>
                            <td>
                                <a href="manage_games.php?edit=<?php echo $game['id']; ?>" class="btn btn-sm btn-warning">Edit</a>
                                <a href="manage_games.php?delete=<?php echo $game['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure?');">Delete</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if (empty($games)): ?>
                            <tr><td colspan="4" class="text-center text-muted">No games found.</td></tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php include '../views/footer.php'; ?>