<?php
// public/register.php

require_once '../config/database.php';

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $college = trim($_POST['college'] ?? '');
    $year = trim($_POST['year'] ?? '');
    $department = trim($_POST['department'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $password = $_POST['password'] ?? '';

    // Validation
    if (empty($name) || empty($college) || empty($year) || empty($department) || empty($phone) || empty($password)) {
        $error = "All fields are required.";
    } elseif (!ctype_digit($phone)) {
        $error = "Phone number must contain digits only.";
    } elseif (strlen($password) < 6) {
        $error = "Password must be at least 6 characters.";
    } else {
        // Generate Unique ID: Random 4 alphanumeric characters (Ensure uniqueness)
        do {
            $unique_id = substr(str_shuffle("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"), 0, 4);
            $checkStmt = $pdo->prepare("SELECT id FROM participants WHERE unique_id = ?");
            $checkStmt->execute([$unique_id]);
        } while ($checkStmt->fetch());
        
        // Hash Password
        $password_hash = password_hash($password, PASSWORD_DEFAULT);

        try {
            $stmt = $pdo->prepare("INSERT INTO participants (unique_id, name, college, year, department, phone, password_hash) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$unique_id, $name, $college, $year, $department, $phone, $password_hash]);

            // Set session and redirect
            $_SESSION['unique_id'] = $unique_id;
            header("Location: profile.php");
            exit;
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) { // Duplicate entry
                $error = "Registration failed. Please try again.";
            } else {
                $error = "Database error: " . $e->getMessage();
            }
        }
    }
}

include '../views/header.php';
?>

<div class="row justify-content-center">
    <div class="col-md-8 col-lg-6">
        <div class="card shadow-soft">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0 h5">Student Registration</h3>
            </div>
            <div class="card-body">
                <?php if ($error): ?>
                    <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <form action="register.php" method="POST">
                    <div class="mb-3">
                        <label for="name" class="form-label">Full Name</label>
                        <input type="text" class="form-control form-control-lg" id="name" name="name" value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>" required>
                    </div>

                    <div class="mb-3">
                        <label for="college" class="form-label">College Name</label>
                        <input type="text" class="form-control form-control-lg" id="college" name="college" value="<?php echo htmlspecialchars($_POST['college'] ?? ''); ?>" required>
                    </div>

                    <div class="mb-3">
                        <label for="year" class="form-label">Year of Study</label>
                        <select class="form-select" id="year" name="year" required>
                            <option value="">Select Year</option>
                            <option value="1" <?php echo (($_POST['year'] ?? '') == '1') ? 'selected' : ''; ?>>1st Year</option>
                            <option value="2" <?php echo (($_POST['year'] ?? '') == '2') ? 'selected' : ''; ?>>2nd Year</option>
                            <option value="3" <?php echo (($_POST['year'] ?? '') == '3') ? 'selected' : ''; ?>>3rd Year</option>
                            <option value="4" <?php echo (($_POST['year'] ?? '') == '4') ? 'selected' : ''; ?>>4th Year</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="department" class="form-label">Department</label>
                        <input type="text" class="form-control form-control-lg" id="department" name="department" placeholder="e.g. Computer Science" value="<?php echo htmlspecialchars($_POST['department'] ?? ''); ?>" required>
                    </div>

                    <div class="mb-3">
                        <label for="phone" class="form-label">Phone Number</label>
                        <input type="tel" class="form-control form-control-lg" id="phone" name="phone" pattern="[0-9]+" title="Digits only" value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>" required>
                    </div>

                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" class="form-control form-control-lg" id="password" name="password" required>
                        <div class="form-text">Minimum 6 characters.</div>
                    </div>

                    <div class="d-grid mt-4">
                        <button type="submit" class="btn btn-primary">Register</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php include '../views/footer.php'; ?>
