<?php
require_once '../config/database.php';

// Access Control
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}

// Fetch Games for Dropdown
if (($_SESSION['role'] ?? '') === 'super') {
    $games = $pdo->query("SELECT * FROM games ORDER BY game_name ASC")->fetchAll();
} else {
    // Fetch only assigned game for regular admin
    $stmt = $pdo->prepare("SELECT g.* FROM games g JOIN admins a ON g.id = a.game_id WHERE a.id = ?");
    $stmt->execute([$_SESSION['admin_id']]);
    $games = $stmt->fetchAll();
}

include '../views/header.php';
?>
<link rel="stylesheet" href="css/scan.css">
<div>
    <div class="row justify-content-center">
        <div class="col-md-8 col-lg-6">
            <div class="card shadow-soft">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0 h5">Assign Points</h4>
                </div>
                <div class="card-body">
                    
                    <!-- Visual Indicator -->
                    <?php if (($_SESSION['role'] ?? '') === 'super'): ?>
                        <div class="alert alert-warning py-2 mb-3 shadow-sm border-0">
                            <small class="fw-bold text-uppercase">⚡ Super Admin Mode</small> &bull; Select any game below.
                        </div>
                    <?php elseif (!empty($games)): ?>
                        <div class="alert alert-info py-2 mb-3 shadow-sm border-0">
                            <span class="fw-bold">🎮 Managing:</span> <?php echo htmlspecialchars($games[0]['game_name']); ?>
                        </div>
                    <?php endif; ?>

                    <!-- Live Counter -->
                    <div id="daily-stats" class="alert alert-light border shadow-sm py-2 mb-3 d-flex justify-content-between align-items-center" style="display:none;">
                        <span class="fw-bold text-muted">📊 Scans Today:</span>
                        <span class="badge bg-primary rounded-pill fs-6" id="scan-count">0</span>
                    </div>

                    <!-- Leaderboard Widget -->
                    <div id="game-leaderboard" class="card border shadow-sm mb-3" style="display:none; background-color: #f8f9fa;">
                        <div class="card-body p-2">
                            <h6 class="card-title text-center text-muted mb-2 small fw-bold text-uppercase">🏆 Top 3 Leaders</h6>
                            <ul class="list-group list-group-flush bg-transparent" id="leaderboard-list"></ul>
                        </div>
                    </div>

                    <!-- Alert Area -->
                    <div id="alert-area"></div>

                    <!-- Settings Form -->
                    <div class="mb-3">
                        <label class="form-label">Select Game</label>
                        <select id="game_id" class="form-select">
                            <?php foreach ($games as $game): ?>
                                <option value="<?php echo $game['id']; ?>"><?php echo htmlspecialchars($game['game_name']); ?> (Max: <?php echo $game['max_points']; ?>)</option>
                            <?php endforeach; ?>
                            <?php if(empty($games)): ?>
                                <option value="">No game assigned. Contact Super Admin.</option>
                            <?php endif; ?>
                        </select>
                    </div>

                    <div class="row">
                        <div class="col-6 mb-3">
                            <label class="form-label">Points</label>
                            <input type="number" id="points" class="form-control" value="10">
                        </div>
                        <div class="col-6 mb-3">
                            <label class="form-label">Note (Opt)</label>
                            <input type="text" id="note" class="form-control" placeholder="e.g. Winner">
                        </div>
                    </div>

                    <hr>

                    <!-- Tabs for Scan vs Manual -->
                    <ul class="nav nav-tabs mb-3" id="myTab" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="scan-tab" data-bs-toggle="tab" data-bs-target="#scan-pane" type="button">Scan QR</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="manual-tab" data-bs-toggle="tab" data-bs-target="#manual-pane" type="button">Manual Entry</button>
                        </li>
                    </ul>

                    <div class="tab-content" id="myTabContent">
                        <!-- Scanner Tab -->
                        <div class="tab-pane fade show active" id="scan-pane" role="tabpanel">
                            <div id="reader"></div>
                            <p class="text-center text-muted mt-3 small">Ensure good lighting for QR scanning.</p>
                        </div>
                        
                        <!-- Manual Entry Tab -->
                        <div class="tab-pane fade" id="manual-pane" role="tabpanel">
                            <div class="input-group mb-3">
                                <input type="text" id="manual_code" class="form-control" placeholder="Enter Unique ID (e.g. A93K)">
                                <button class="btn btn-success" type="button" id="btn-manual-submit">Submit</button>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://unpkg.com/html5-qrcode" type="text/javascript"></script>

<script>
    function assignPoints(uniqueId) {
        const gameId = document.getElementById('game_id').value;
        const points = document.getElementById('points').value;
        const note = document.getElementById('note').value;
        const alertArea = document.getElementById('alert-area');

        if (!gameId) {
            alert("Please select a game first.");
            return;
        }

        fetch('assign_points.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ unique_id: uniqueId, game_id: gameId, points: points, note: note })
        })
        .then(response => response.text()) // Get text first to debug if not JSON
        .then(text => {
            try {
                return JSON.parse(text);
            } catch (e) {
                console.error("Server Error:", text);
                throw new Error("Invalid server response");
            }
        })
        .then(data => {
            const type = data.success ? 'success' : 'danger';
            let undoBtn = '';
            if (data.success && data.log_id) {
                undoBtn = ` <button class="btn btn-sm btn-link text-decoration-none fw-bold text-reset" onclick="undoLastAction(${data.log_id})">Undo</button>`;
            }
            alertArea.innerHTML = `<div class="alert alert-${type} alert-dismissible fade show" role="alert">
                                    ${data.message}${undoBtn}
                                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                                   </div>`;
            if (data.success) {
                fetchScanCount();
                fetchLeaderboard();
            }
        })
        .catch(err => console.error(err));
    }

    // Safe Audio Context
    let audioCtx = null;
    try {
        audioCtx = new (window.AudioContext || window.webkitAudioContext)();
    } catch (e) {
        console.warn("AudioContext not supported");
    }

    function playBeep() {
        if (!audioCtx) return;
        if (audioCtx.state === 'suspended') audioCtx.resume();

        const oscillator = audioCtx.createOscillator();
        const gainNode = audioCtx.createGain();

        oscillator.connect(gainNode);
        gainNode.connect(audioCtx.destination);

        oscillator.type = "sine";
        oscillator.frequency.value = 1000;
        gainNode.gain.value = 0.1;

        oscillator.start();
        setTimeout(() => { oscillator.stop(); }, 200);
    }

    let lastScannedCode = null;

    function onScanSuccess(decodedText, decodedResult) {
        if (decodedText !== lastScannedCode) {
            lastScannedCode = decodedText;
            playBeep();
            assignPoints(decodedText);
            // Prevent double-scanning the same code for 3 seconds
            setTimeout(() => { lastScannedCode = null; }, 3000);
        }
    }

    function onScanFailure(error) {
        // Handle scan failure, usually better to ignore and keep scanning.
    }

    function undoLastAction(logId) {
        if (!confirm("Are you sure you want to undo this assignment?")) return;

        fetch('undo_points.php', { 
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ log_id: logId })
        })
        .then(r => r.json())
        .then(data => {
            const alertArea = document.getElementById('alert-area');
            const type = data.success ? 'warning' : 'danger';
            alertArea.innerHTML = `<div class="alert alert-${type} alert-dismissible fade show" role="alert">
                                    ${data.message}
                                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                                   </div>`;
            if (data.success) {
                fetchScanCount();
                fetchLeaderboard();
            }
        })
        .catch(err => console.error(err));
    }

    // Manual Submit Logic
    const manualInput = document.getElementById('manual_code');
    const manualBtn = document.getElementById('btn-manual-submit');

    function triggerManualSubmit() {
        const code = manualInput.value.trim();
        if (code) {
            assignPoints(code);
            manualInput.value = ''; // Clear input
        }
    }

    if (manualBtn) manualBtn.addEventListener('click', triggerManualSubmit);
    if (manualInput) manualInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') triggerManualSubmit();
    });

    // Initialize QR Scanner
    if (typeof Html5QrcodeScanner !== 'undefined') {
        const html5QrcodeScanner = new Html5QrcodeScanner("reader", { fps: 10, qrbox: 250 });
        html5QrcodeScanner.render(onScanSuccess, onScanFailure);
    }

    // Live Counter Logic
    const gameSelect = document.getElementById('game_id');
    const statsBox = document.getElementById('daily-stats');
    const countSpan = document.getElementById('scan-count');
    const leaderboardBox = document.getElementById('game-leaderboard');
    const leaderboardList = document.getElementById('leaderboard-list');

    function fetchScanCount() {
        const gameId = gameSelect ? gameSelect.value : null;
        if (!gameId) {
            if (statsBox) statsBox.style.display = 'none';
            return;
        }
        fetch(`get_game_stats.php?game_id=${gameId}`)
            .then(r => r.json())
            .then(data => {
                if (statsBox) statsBox.style.display = 'flex';
                if (countSpan) countSpan.innerText = data.count;
            });
    }

    function fetchLeaderboard() {
        const gameId = gameSelect ? gameSelect.value : null;
        if (!gameId) {
            if (leaderboardBox) leaderboardBox.style.display = 'none';
            return;
        }
        
        fetch(`get_game_leaderboard.php?game_id=${gameId}`)
            .then(r => r.json())
            .then(data => {
                if (data.length > 0 && leaderboardBox && leaderboardList) {
                    leaderboardBox.style.display = 'block';
                    leaderboardList.innerHTML = '';
                    data.forEach((p, index) => {
                        let badgeColor = index === 0 ? 'warning' : (index === 1 ? 'secondary' : 'danger');
                        let icon = index === 0 ? '🥇' : (index === 1 ? '🥈' : '🥉');
                        let textColor = index === 0 ? 'text-dark' : 'text-white';
                        
                        let li = document.createElement('li');
                        li.className = 'list-group-item d-flex justify-content-between align-items-center bg-transparent px-2 py-1 border-0';
                        li.innerHTML = `<span class="small text-truncate" style="max-width: 200px;">${icon} <strong>${p.name}</strong></span>
                                        <span class="badge bg-${badgeColor} rounded-pill ${textColor}">${p.points}</span>`;
                        leaderboardList.appendChild(li);
                    });
                } else {
                    if (leaderboardBox) leaderboardBox.style.display = 'none';
                }
            });
    }

    if (gameSelect) {
        gameSelect.addEventListener('change', () => { fetchScanCount(); fetchLeaderboard(); });
        fetchScanCount(); // Initial load
        fetchLeaderboard();
    }
</script>
<?php include '../views/footer.php'; ?>